% VA_Shares_Components_Trends.m
% 
% Compute Trend Values of Matrices and Value-added Ratios
% 
% "The Past and Future of U.S. Structural Change" 
% Andrew Foerster, Andreas Hornstein, Pierre-Daniel Sarte, Mark Watson
% September 2025
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % 

% -- Clear Workspace -- %
restoredefaultpath;
clear;
clc;
close all;

% -- File Directories -- % 
datadir = 'Data\DataReplication\DataFinal\'; 
outdir  = 'Tables\';
figdir  = 'Figures\';
matdir  = 'Output\';

% -- Add paths -- %
addpath('Functions\');
addpath('Utilities\')
addpath('Data\')

% -- Load Data and Setup -- %
load_data;
period_lower    = 20;       % Shortest Period 
Share_mat       = sV_mat;
T               = size(Share_mat,1);
nS              = size(Share_mat,2);
SecName_1       = SecName;
i_base          = 5;  % Base sector for logit transformation

% Compute q
q       = floor(2*T/period_lower);
per_q   = 2*T/q;
fprintf('Number of cosine transformations for in-sample period: %d\n',q);
fprintf('Number of Shortest Periods: %5.1f \n',per_q);

% -- Get the trends for the various matrices, etc. --%
% Paramter Value
i_trend = 1;  % Use the model with linear trends throughout

% Construct Trends for shares
% VA Shares
va_shares = Share_mat;
va_shares_trend = share_trend(va_shares,q,i_base,i_trend);

% Consumption Shares
theta_trend = share_trend(theta,q,i_base,i_trend);

% Columns of phi
phi_trend = NaN(size(phi));
for i = 1:nS
    tmp = squeeze(phi(:,i,:));
    tmp = tmp';
    tmp_proj = share_trend(tmp,q,i_base,i_trend);
    phi_trend(:,i,:) = tmp_proj';
end

% Columns of omega
omega_trend = NaN(size(omega));
for i = 1:nS
    tmp = squeeze(omega(:,i,:));
    tmp = tmp';
    tmp_proj = share_trend(tmp,q,i_base,i_trend);
    omega_trend(:,i,:) = tmp_proj';
end

% gamma ... note that gamma is between zero and 1 ... so impose this on the trends
gamma_trend = NaN(size(gamma));
for i = 1:nS
    tmp = [gamma(:,i) 1-gamma(:,i)];
    tmp_proj = share_trend(tmp,q,2,i_trend);
    gamma_trend(:,i) = tmp_proj(:,1);
end

% gamma ... note that gamma is between zero and 1 ... so impose this on the trends
gamma_trend = NaN(size(gamma));
for i = 1:nS
    tmp = [gamma(:,i) 1-gamma(:,i)];
    tmp_proj = share_trend(tmp,q,2,i_trend);
    gamma_trend(:,i) = tmp_proj(:,1);
end

% psiX ... note that psiX is between zero and 1 ... so impose this on the trends
psiX_trend = NaN(size(psiX));
for i = 1:nS
    tmp = [psiX(:,i) 1-psiX(:,i)];
    tmp_proj = share_trend(tmp,q,2,i_trend);
    psiX_trend(:,i) = tmp_proj(:,1);
end

% psiNX and psiG ... do this unconstrained
psiNX_trend = NaN(size(psiNX));
psiG_trend = NaN(size(psiG));
for i = 1:nS
    psiNX_trend(:,i) = LF_Proj(psiNX(:,i),q,i_trend);
    psiG_trend(:,i) = LF_Proj(psiG(:,i),q,i_trend);
end


% Save Results
save([matdir 'VA_Shares_Components_Trends.mat'], ...
    'va_shares','va_shares_trend','theta','theta_trend',...
    'phi','phi_trend','omega','omega_trend','gamma','gamma_trend','psiX','psiX_trend',...
    'psiG','psiG_trend','psiNX','psiNX_trend','calvec','SecName');

% ------------------------------ Functions ----------------------------

function s_proj = share_trend(share_actual,q,i_base,i_trend)
% Construct trend values for a matrix of shares

    % Compute logit transformation
    X = logit_s_to_x(share_actual,i_base);
    Y = X(:,1:end ~= i_base);

    % Get some results from I_1 model
    Yproj = LF_Proj(Y,q,i_trend);
    s_proj = logit_y_to_s(Yproj,i_base);

end

